﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/

using System;
using System.Windows.Forms;
using Inet.Viewer.Data;

namespace Inet.Viewer.WinForms
{
    /// <summary>
    /// Representing a toolbar item which shows a progress. It automaticly updates its UI elements according to the current 
    /// value of the progress instance.
    /// </summary>
    public class ToolStripProgress : ToolStripControlHost
    {
        private Progress progress;
        /// <summary>
        /// Event will be triggered when progress reaches status finished.
        /// </summary>
        public event EventHandler Finished;

        /// <summary>
        /// Creates a toolbar item for the specified progress instance. Registers itselfs to the change events of the progress.
        /// </summary>
        /// <param name="progress">the progress to show</param>
        public ToolStripProgress(Progress progress) :
            base(new ProgressControl())
        {
            this.progress = progress;
            progress.ProgressChanged += ProgressChanged;
            progress.StatusChanged += StatusChanged;

            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(ViewerToolbar));
            ProgressControl control = (ProgressControl)Control;
            control.ProgressName = progress.Name;
            control.StopClicked += StopButtonClick;
        }

        /// <summary>
        /// Called on ProgressChanged events. Updates the UI elements.
        /// </summary>
        /// <param name="sender">the sender of the event</param>
        /// <param name="e">the event's arguments</param>
        private void ProgressChanged(object sender, EventArgs e)
        {
            ProgressControl control = (ProgressControl)Control;
            if (control.InvokeRequired)
            {
                control.BeginInvoke(new Action(delegate
                {
                    UpdateProgress(control);
                }));
            }
            else
            {
                UpdateProgress(control);
            }
        }

        /// <summary>
        /// Updates the ProgressMode, name and the value of the ProgressControl
        /// </summary>
        /// <param name="control">The ProgressControl, that should be updated</param>
        private void UpdateProgress(ProgressControl control)
        {
            control.ProgressMode = progress.ProgressMode;
            control.ProgressName = progress.Name;
            if (progress.TotalProgress != 0)
            {
                control.ProgressValue = Math.Min(100, 100 * progress.ProgressCount / progress.TotalProgress);
            }
        }

        /// <summary>
        /// Called on StatusChanged events. If the progress is finished an event
        /// is forwarded to the listeners of the Finished handler.
        /// </summary>
        /// <param name="sender">the sender of the event</param>
        /// <param name="e">the event's arguments</param>
        private void StatusChanged(object sender, EventArgs e)
        {
            if (progress.Finished)
            {
                if (Finished != null)
                {
                    Finished(this, new EventArgs());
                }
            }
        }

        /// <summary>
        /// Called when the user clicks on the stop button. Cancels the progress.
        /// </summary>
        /// <param name="sender">the sender of the event</param>
        /// <param name="e">the event's arguments</param>
        private void StopButtonClick(object sender, EventArgs e)
        {
            progress.Cancel();
        }

        /// <summary>
        /// Gets the progress instance.
        /// </summary>
        public Progress Progress
        {
            get
            {
                return progress;
            } 
        }
    }
}
